/*
** Zabbix
** Copyright (C) 2001-2016 Zabbix SIA
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

#include "sysinc.h"
#include "module.h"
#include "common.h"
#include "zbxexec.h"
#include "log.h"
#include "ve_item.h"

/* the variable keeps timeout setting for item processing */
static int	item_timeout = 0;

#define BINPATH "/opt/nec/ve/bin/"
#define VECMD "/opt/nec/ve/bin/vecmd"

extern int parse_config_file (void);
extern char *veos_bindir;

ve_path_t	ve_path[VE_MAX] =
{
{"veslot0", {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}, "", ""},
{"veslot1", {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}, "", ""},
{"veslot2", {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}, "", ""},
{"veslot3", {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}, "", ""},
{"veslot4", {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}, "", ""},
{"veslot5", {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}, "", ""},
{"veslot6", {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}, "", ""},
{"veslot7", {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}, "", ""},
};

int	zbx_veosmodule_state(AGENT_REQUEST *request, AGENT_RESULT *result);
int	zbx_veosmodule_memory_size(AGENT_REQUEST *request, AGENT_RESULT *result);
int	zbx_veosmodule_ve_load(AGENT_REQUEST *request, AGENT_RESULT *result);
int	zbx_veosmodule_ve_util(AGENT_REQUEST *request, AGENT_RESULT *result);

static ZBX_METRIC keys[] =
/*      KEY                     FLAG		FUNCTION        	TEST PARAMETERS */
{
	{"ve_os.state",	CF_HAVEPARAMS,	zbx_veosmodule_state,	"AuroraVE-0000-0, 0"},
	{"ve_vm.memory.size",	CF_HAVEPARAMS,	zbx_veosmodule_memory_size,	"AuroraVE-0000-0, 0, total"},
	{"ve_system.ve.load",	CF_HAVEPARAMS,	zbx_veosmodule_ve_load,	"AuroraVE-0000-0, 0, all, avg5"},
	{"ve_system.ve.util",	CF_HAVEPARAMS,	zbx_veosmodule_ve_util,	"AuroraVE-0000-0, 0, 1, idle"},
	{NULL}
};

/******************************************************************************
 *                                                                            *
 * Function: zbx_module_api_version                                           *
 *                                                                            *
 * Purpose: returns version number of the module interface                    *
 *                                                                            *
 * Return value: ZBX_MODULE_API_VERSION_ONE - the only version supported by   *
 *               Zabbix currently                                             *
 *                                                                            *
 ******************************************************************************/
int	zbx_module_api_version()
{
	return ZBX_MODULE_API_VERSION_ONE;
}

/******************************************************************************
 *                                                                            *
 * Function: zbx_module_item_timeout                                          *
 *                                                                            *
 * Purpose: set timeout value for processing of items                         *
 *                                                                            *
 * Parameters: timeout - timeout in seconds, 0 - no timeout set               *
 *                                                                            *
 ******************************************************************************/
void	zbx_module_item_timeout(int timeout)
{
	item_timeout = timeout;
}

/******************************************************************************
 *                                                                            *
 * Function: zbx_module_item_list                                             *
 *                                                                            *
 * Purpose: returns list of item keys supported by the module                 *
 *                                                                            *
 * Return value: list of item keys                                            *
 *                                                                            *
 ******************************************************************************/
ZBX_METRIC	*zbx_module_item_list()
{
	return keys;
}

static	char	*
get_vepath(char *vnp, AGENT_RESULT *result)
{
	int	vn, rv, l;
	char	*dp;
	char	cmdbuf[BUFSIZ], errbuf[BUFSIZ];
	char	*obp = NULL;
        unsigned char flag = ZBX_EXIT_CODE_CHECKS_ENABLED;

	/* there is no strict validation of parameters for simplicity sake */
	vn = strtoul(vnp, &dp, 0);
	if (dp == vnp
	 || *dp != '\0'
	 || vn >= VE_MAX)
	{
		SET_MSG_RESULT(result, strdup("Invalid value given"));
		return(NULL);
	}
	if (ve_path[vn].path[0] != '\0') {
		return(ve_path[vn].path);
    }
	if (ve_path[vn].slot[0] == '\0') {
		SET_MSG_RESULT(result, strdup("device node not found"));
		return(NULL);
	}
	zbx_snprintf(cmdbuf, BUFSIZ, "/usr/bin/udevadm info --query=path --name=%s 2>/dev/null", ve_path[vn].slot);
	rv = zbx_execute(cmdbuf, &obp, errbuf, BUFSIZ, item_timeout, flag, NULL);
	switch (rv) {
        case SUCCEED:
            zabbix_log(LOG_LEVEL_DEBUG, "%s", obp);
            if (*obp != '/')
            {
                SET_MSG_RESULT(result, strdup(obp));
                zbx_free(obp);
                return(NULL);
            }
            l = strlen(obp);
            if (obp[l - 1] == '\n')
                obp[--l] = '\0';
            zbx_snprintf(ve_path[vn].path, PATH_MAX, "/sys%s", obp);
            zbx_free(obp);
            break;
        case TIMEOUT_ERROR:
            SET_MSG_RESULT(result, strdup("udevadm execution timeout"));
            return(NULL);
        case FAIL:
            zabbix_log(LOG_LEVEL_DEBUG, "%s", errbuf);
            SET_MSG_RESULT(result, strdup(errbuf));
            return(NULL);
	}

	return(ve_path[vn].path);
}

static int os_state_is_available (char *vnp, AGENT_RESULT *result)
{
	char	*vpp;
	FILE	*vfd;
	char	valpath[BUFSIZ], valbuf[BUFSIZ];

	vpp = get_vepath(vnp, result);
	if (vpp == NULL) {
		return -1;
    }

	zbx_snprintf(valpath, BUFSIZ, "%s/%s", vpp, "os_state");

	vfd = fopen(valpath, "r");
	if (vfd == NULL) {
		SET_MSG_RESULT(result, strdup("File(os_state) can not access"));
		return -1;
	}
	if (fgets(valbuf, BUFSIZ, vfd) == NULL) {
		SET_MSG_RESULT(result, strdup("File(os_state) can not access"));
		fclose(vfd);
		return -1;
	}
	fclose(vfd);
	valbuf[1] = '\0';

    if (valbuf[0] == '0') {
        return 1; /* available */
    } else {
		SET_MSG_RESULT(result, strdup("veos is unavailable."));
        return 0; /* unavailable */
    }
}

/******************************************************************************
 *                                                                            *
 * Function: zbx_vemodule_metric                                              *
 *                                                                            *
 * Purpose: a main entry point for processing of an item                      *
 *                                                                            *
 * Parameters: request - structure that contains item key and parameters      *
 *              request->key - item key without parameters                    *
 *              request->nparam - number of parameters                        *
 *              request->timeout - processing should not take longer than     *
 *                                 this number of seconds                     *
 *              request->params[N-1] - pointers to item key parameters        *
 *                                                                            *
 *             result - structure that will contain result                    *
 *                                                                            *
 * Return value: SYSINFO_RET_FAIL - function failed, item will be marked      *
 *                                 as not supported by zabbix                 *
 *               SYSINFO_RET_OK - success                                     *
 *                                                                            *
 * Comment: get_rparam(request, N-1) can be used to get a pointer to the Nth  *
 *          parameter starting from 0 (first parameter). Make sure it exists  *
 *          by checking value of request->nparam.                             *
 *                                                                            *
 ******************************************************************************/

int	zbx_veosmodule_state(AGENT_REQUEST *request, AGENT_RESULT *result)
{
	char	*vnp, *vpp;
	FILE	*vfd;
	char	valpath[BUFSIZ], valbuf[BUFSIZ];

	if (request->nparam != 1) {
		/* set optional error message */
		SET_MSG_RESULT(result, strdup("Invalid number of parameters"));
		return SYSINFO_RET_FAIL;
	}

	vnp = get_rparam(request, 0);
	vpp = get_vepath(vnp, result);
	if (vpp == NULL) {
		return SYSINFO_RET_FAIL;
    }
	zbx_snprintf(valpath, BUFSIZ, "%s/%s", vpp, "os_state");
	vfd = fopen(valpath, "r");
	if (vfd == NULL) {
		SET_MSG_RESULT(result, strdup("File(os_state) can not access"));
		return SYSINFO_RET_FAIL;
	}
	if (fgets(valbuf, BUFSIZ, vfd) == NULL) {
		SET_MSG_RESULT(result, strdup("File(os_state) can not access"));
		fclose(vfd);
		return SYSINFO_RET_FAIL;
	}
	fclose(vfd);
	valbuf[1] = '\0';
	SET_STR_RESULT(result, strdup(valbuf));
	return SYSINFO_RET_OK;
}

static	int64_t
veos_memory_total(char *dirp, AGENT_RESULT *result)
{
	int64_t	msz;
	char	valpath[BUFSIZ], valbuf[BUFSIZ], *dp;
	FILE	*vfd;

	zbx_snprintf(valpath, BUFSIZ, "%s/memory_size", dirp);
	vfd = fopen(valpath, "r");
	if (vfd == NULL) {
		SET_MSG_RESULT(result, strdup("File(memory_size) can not access"));
		return -1;
	}
	if (fgets(valbuf, BUFSIZ, vfd) == NULL) {
		SET_MSG_RESULT(result, strdup("File(memory_size) can not access"));
		fclose(vfd);
		return -1;
	}
	fclose(vfd);
	msz = strtoul(valbuf, &dp, 0);
	if (msz == 0) {
		SET_MSG_RESULT(result, strdup("memory_size is invalid"));
		return -1;
	}
	msz *= 0x40000000;	/* GB -> B */
	return msz;
}

static	int64_t
veos_memory(int ven, int mode, AGENT_RESULT *result)
{
	int64_t	fsz, freemem = -1, buffer = -1, cached = -1;
	int	rv;
	char	*vp, *dp;
	char	cmdbuf[80], errbuf[BUFSIZ];
	char	*obp = NULL;
        unsigned char flag = ZBX_EXIT_CODE_CHECKS_ENABLED;

	zbx_snprintf(cmdbuf, 80, "env VE_NODE_NUMBER=%d %s/ve-vmstat -S K | tail -1", ven, veos_bindir);
	rv = zbx_execute(cmdbuf, &obp, errbuf, BUFSIZ, item_timeout, flag, NULL);
	switch (rv) {
        case SUCCEED:
            zabbix_log(LOG_LEVEL_DEBUG, "%s", obp);
            strtok(obp, " ");
            break;
        case TIMEOUT_ERROR:
            SET_MSG_RESULT(result, strdup("ve-vmstat execution timeout"));
            return(-1);
        case FAIL:
            zabbix_log(LOG_LEVEL_DEBUG, "%s", errbuf);
            SET_MSG_RESULT(result, strdup(errbuf));
            return(-1);
        default:
            zabbix_log(LOG_LEVEL_DEBUG, "Unknown error occurred (rv: %d)", rv);
            SET_MSG_RESULT(result, strdup("Unknown error occurred."));
            return -1;
	}

    switch (mode)  {
        case 2: /* available" */
        case 4: /* cached */
            strtok(NULL, " ");
        case 3: /* buffer */
            strtok(NULL, " ");
        case 1: /* free */
            vp = strtok(NULL, " ");
            if (vp == NULL) {
                goto fail;
            }
            freemem = strtoul(vp, &dp, 0);

            vp = strtok(NULL, " ");
            if (vp == NULL) {
                goto fail;
            }
            buffer = strtoul(vp, &dp, 0);
            
            vp = strtok(NULL, " ");
            if (vp == NULL) {
                goto fail;
            }
            fsz = cached = strtoul(vp, &dp, 0);
            break;
    }
    
    if (mode == 2) {
        fsz = freemem + buffer + cached;
    }

	fsz *= 1024;
	zbx_free(obp);
	return(fsz);

fail:
    SET_MSG_RESULT(result, strdup("Invalid ve-vmstat format."));
    zbx_free(obp);
    return -1;
}

/*
 * ve_vm.memory.size[host,X,mode]
 * return value: UINTBUFSIZ(MB)
 */
int	zbx_veosmodule_memory_size(AGENT_REQUEST *request, AGENT_RESULT *result)
{
	int64_t	msz, fsz;
	double	pfr;
	char	*vnp, *vpp, *mdp, *dp;
	int	vn, mod;
	char	dirpath[BUFSIZ];

	if (request->nparam != 1
	 && request->nparam != 2) {
		/* set optional error message */
		SET_MSG_RESULT(result, strdup("Invalid number of parameters"));
		return SYSINFO_RET_FAIL;
	}

	vnp = get_rparam(request, 0);

    if (os_state_is_available (vnp, result) <= 0) {
        return SYSINFO_RET_FAIL;
    }

	vpp = get_vepath(vnp, result);
	if (vpp == NULL) {
		return SYSINFO_RET_FAIL;
    }
	vn = strtoul(vnp, &dp, 0);

	if (request->nparam == 1) {
		mod = 0;
    } else {
		mdp = get_rparam(request, 1);
		if (strcmp(mdp, "total") == 0) {
			mod = 0;
        } else if (strcmp(mdp, "free") == 0) {
			mod = 1;
        } else if (strcmp(mdp, "available") == 0) {
			mod = 2;
        } else if (strcmp(mdp, "buffers") == 0) {
			mod = 3;
        } else if (strcmp(mdp, "cached") == 0) {
			mod = 4;
        } else if (strcmp(mdp, "pavailable") == 0) {
			mod = 5;
        } else {
			/* set optional error message */
			SET_MSG_RESULT(result, strdup("Invalid mode given"));
			return SYSINFO_RET_FAIL;
		}
	}

    zbx_snprintf(dirpath, BUFSIZ, "%s", vpp);

	switch (mod) {
        case 0: /* total */
            msz = veos_memory_total(dirpath, result);
            if (msz < 0) {
                return SYSINFO_RET_FAIL;
            }
            SET_UI64_RESULT(result, msz);
            break;
        case 1: /* free */
        case 2: /* available" */
        case 3: /* buffer */
        case 4: /* cached */
            fsz = veos_memory(vn, mod, result);
            if (fsz < 0) {
                return SYSINFO_RET_FAIL;
            }
            SET_UI64_RESULT(result, fsz);
            break;
        case 5: /* pavailable */
            msz = veos_memory_total(dirpath, result);
            if (msz < 0) {
                return SYSINFO_RET_FAIL;
            }
            fsz = veos_memory(vn, 2, result); /* Get available */
            if (fsz < 0) {
                return SYSINFO_RET_FAIL;
            }
            pfr = (double)(fsz * 100) / (double)msz;
            SET_DBL_RESULT(result, pfr);
            break;
    }

	return SYSINFO_RET_OK;
}

static	int
veos_num_core(char *dirp, AGENT_RESULT *result)
{
	int	cores;
	char	valpath[BUFSIZ], valbuf[BUFSIZ], *dp;
	FILE	*vfd;

	zbx_snprintf(valpath, BUFSIZ, "%s/num_of_core", dirp);
	vfd = fopen(valpath, "r");
	if (vfd == NULL) {
		SET_MSG_RESULT(result, strdup("File(num_of_core) can not access"));
		return -1;
	}
	if (fgets(valbuf, BUFSIZ, vfd) == NULL) {
		SET_MSG_RESULT(result, strdup("File(num_of_core) can not access"));
		fclose(vfd);
		return -1;
	}
	fclose(vfd);
	cores = strtoul(valbuf, &dp, 0);
	if (cores == 0) {
		SET_MSG_RESULT(result, strdup("num_of_core is invalid"));
		return -1;
	}
	return cores;
}

/*
 * ve_system.ve.load[host,X,cpu,mode]
 * return value: double
 */
int	zbx_veosmodule_ve_load(AGENT_REQUEST *request, AGENT_RESULT *result)
{
	double	lod;
	char	*vnp, *vpp, *cpp, *mdp, *vp, *dp;
	int	vn, cpu, mod, cores, rv;
	char	cmdbuf[80], errbuf[BUFSIZ], dirpath[BUFSIZ];
	char	*obp = NULL;
        unsigned char flag = ZBX_EXIT_CODE_CHECKS_ENABLED;

	if (request->nparam < 1
	 || request->nparam > 3) {
		/* set optional error message */
		SET_MSG_RESULT(result, strdup("Invalid number of parameters"));
		return SYSINFO_RET_FAIL;
	}

	vnp = get_rparam(request, 0);

    if (os_state_is_available (vnp, result) <= 0) {
        return SYSINFO_RET_FAIL;
    }

	vpp = get_vepath(vnp, result);
	if (vpp == NULL) {
		return SYSINFO_RET_FAIL;
    }
	vn = strtoul(vnp, &dp, 0);

	if (request->nparam == 1) {
		cpu = 0;
		mod = 0;
	} else {
		cpp = get_rparam(request, 1);
		if (cpp == NULL
		 || *cpp == '\0'
		 || strcmp(cpp, "all") == 0) {
			cpu = 0;
        } else if (strcmp(cpp, "percore") == 0) {
			cpu = 1;
        } else {
			/* set optional error message */
			SET_MSG_RESULT(result, strdup("Invalid cpu given"));
			return SYSINFO_RET_FAIL;
		}
		if (request->nparam == 2) {
			mod = 0;
        } else {
			mdp = get_rparam(request, 2);
			if (strcmp(mdp, "avg1") == 0) {
				mod = 0;
            } else if (strcmp(mdp, "avg5") == 0) {
				mod = 1;
            } else if (strcmp(mdp, "avg15") == 0) {
				mod = 2;
            } else {
				/* set optional error message */
				SET_MSG_RESULT(result, strdup("Invalid mode given"));
				return SYSINFO_RET_FAIL;
			}
		}
	}

	zbx_snprintf(cmdbuf, 80, "env VE_NODE_NUMBER=%d %s/ve-uptime", vn, veos_bindir);

	rv = zbx_execute(cmdbuf, &obp, errbuf, BUFSIZ, item_timeout, flag, NULL);
    switch (rv) {
        case SUCCEED:
            zabbix_log(LOG_LEVEL_DEBUG, "%s", obp);
            vp = strstr(obp, "load average:");
            vp = strtok(vp, ":");
            break;
        case TIMEOUT_ERROR:
            SET_MSG_RESULT(result, strdup("ve-vmstat execution timeout"));
            return SYSINFO_RET_FAIL;
        case FAIL:
            zabbix_log(LOG_LEVEL_DEBUG, "%s", errbuf);
            SET_MSG_RESULT(result, strdup(errbuf));
            return SYSINFO_RET_FAIL;
    }

	switch (mod) {
        case 2: /* avg15 */
            strtok(NULL, " ,\n");
        case 1: /* avg5 */
            strtok(NULL, " ,\n");
        case 0: /* avg1 */
            vp = strtok(NULL, " ,\n");
	}
	lod = strtod(vp, &dp);
	if (cpu == 1) {
        zbx_snprintf(dirpath, BUFSIZ, "%s", vpp);
		if ((cores = veos_num_core(dirpath, result)) < 0) {
            zbx_free(obp);
            return SYSINFO_RET_FAIL;
        }
		lod /= cores;
	}
	SET_DBL_RESULT(result, lod);
	zbx_free(obp);

	return SYSINFO_RET_OK;
}

/*
 * ve_system.ve.util[host,X,core,type]
 * return value: double
 */
int	zbx_veosmodule_ve_util(AGENT_REQUEST *request, AGENT_RESULT *result)
{
	double	utl;
	char	*vnp, *vpp, *cnp, *tpp, *vp, *dp;
	int	vn, cn, typ, rv;
	char	cmdbuf[80], errbuf[BUFSIZ];
	char	*obp = NULL;
        unsigned char flag = ZBX_EXIT_CODE_CHECKS_ENABLED;

	if (request->nparam < 1
	 || request->nparam > 3) {
		/* set optional error message */
		SET_MSG_RESULT(result, strdup("Invalid number of parameters"));
		return SYSINFO_RET_FAIL;
	}

	vnp = get_rparam(request, 0);

    if (os_state_is_available (vnp, result) <= 0) {
        return SYSINFO_RET_FAIL;
    }

	vpp = get_vepath(vnp, result);
	if (vpp == NULL) {
		return SYSINFO_RET_FAIL;
    }
	vn = strtoul(vnp, &dp, 0);

	if (request->nparam == 1) {
		cn = -1;
		typ = 5;
	} else {
		cnp = get_rparam(request, 1);
		if (cnp == NULL
		 || *cnp == '\0'
         || strcmp (cnp, "all") == 0) {
			cn = -1;
        } else {
			cn = strtoul(cnp, &dp, 0);
			if (dp == cnp
			 || *dp != '\0')
			{
				/* set optional error message */
				SET_MSG_RESULT(result, strdup("Invalid cpu given"));
				return SYSINFO_RET_FAIL;
			}
		}
		if (request->nparam == 2) {
			typ = 5;
        } else {
			tpp = get_rparam(request, 2);
			if (strcmp(tpp, "idle") == 0) {
				typ = 0;
            } else if (strcmp(tpp, "softirq") == 0) {
				typ = 1;
            } else if (strcmp(tpp, "interrupt") == 0) {
				typ = 2;
            } else if (strcmp(tpp, "system") == 0) {
				typ = 3;
            } else if (strcmp(tpp, "nice") == 0) {
				typ = 4;
            } else if (strcmp(tpp, "user") == 0) {
				typ = 5;
            } else {
				/* set optional error message */
				SET_MSG_RESULT(result, strdup("Invalid type given"));
				return SYSINFO_RET_FAIL;
			}
		}
	}

	if (cn == -1) {
		zbx_snprintf(cmdbuf, 80, "env LC_ALL=C VE_NODE_NUMBER=%d %s/ve-mpstat | tail -1", vn, veos_bindir);
    } else {
		zbx_snprintf(cmdbuf, 80, "env LC_ALL=C VE_NODE_NUMBER=%d %s/ve-mpstat -P %d | tail -1", vn, veos_bindir, cn);
    }
	rv = zbx_execute(cmdbuf, &obp, errbuf, BUFSIZ, item_timeout, flag, NULL);
    switch (rv) {
        case SUCCEED:
            zabbix_log(LOG_LEVEL_DEBUG, "%s", obp);
            vp = strtok(obp, " ");
            break;
        case TIMEOUT_ERROR:
            SET_MSG_RESULT(result, strdup("ve-vmstat execution timeout"));
            return SYSINFO_RET_FAIL;
        case FAIL:
            zabbix_log(LOG_LEVEL_DEBUG, "%s", errbuf);
            SET_MSG_RESULT(result, strdup(errbuf));
            return SYSINFO_RET_FAIL;
    }

	switch (typ) {
	case 0: /* idle */
		strtok(NULL, " \n");
		strtok(NULL, " \n");
		strtok(NULL, " \n");
		strtok(NULL, " \n");
    case 1: /* softirq */
		strtok(NULL, " \n");
    case 2: /* interrupt */
		strtok(NULL, " \n");
		strtok(NULL, " \n");
	case 3: /* system */
		strtok(NULL, " \n");
    case 4: /* nice */
		strtok(NULL, " \n");
	case 5: /* user */
		strtok(NULL, " \n");
		vp = strtok(NULL, " \n");
	}
	utl = strtod(vp, &dp);
	SET_DBL_RESULT(result, utl);
	zbx_free(obp);

	return SYSINFO_RET_OK;
}


/******************************************************************************
 *                                                                            *
 * Function: zbx_module_init                                                  *
 *                                                                            *
 * Purpose: the function is called on agent startup                           *
 *          It should be used to call any initialization routines             *
 *                                                                            *
 * Return value: ZBX_MODULE_OK - success                                      *
 *               ZBX_MODULE_FAIL - module initialization failed               *
 *                                                                            *
 * Comment: the module won't be loaded in case of ZBX_MODULE_FAIL             *
 *                                                                            *
 ******************************************************************************/
int	zbx_module_init()
{
    parse_config_file();
	zabbix_log(LOG_LEVEL_DEBUG, "ve_os_item init.");

	return ZBX_MODULE_OK;
}

/******************************************************************************
 *                                                                            *
 * Function: zbx_module_uninit                                                *
 *                                                                            *
 * Purpose: the function is called on agent shutdown                          *
 *          It should be used to cleanup used resources if there are any      *
 *                                                                            *
 * Return value: ZBX_MODULE_OK - success                                      *
 *               ZBX_MODULE_FAIL - function failed                            *
 *                                                                            *
 ******************************************************************************/
int	zbx_module_uninit()
{
	zabbix_log(LOG_LEVEL_DEBUG, "ve_os_item uninit.");
	return ZBX_MODULE_OK;
}

