#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <yaml.h>

#include "zbxcommon.h"
#include "zbxcfg.h"
#include "zbxlog.h"

#define MAX_LEN     32
#define MAP_TOP     1
#define MAP_VE      2
#define MAP_SENSOR  3
#define MAP_SEARCH  4
#define SENSOR_INFO "sensor_info"
#define SYSFS_KEY "sysfs_file"

#define CONFIG "/etc/opt/nec/zabbix/ve.conf"
#define DEFAULT_VECMD_BINDIR    "/opt/nec/ve/bin"
#define DEFAULT_VEOS_BINDIR     "/opt/nec/ve/bin"
#define DEFAULT_SENSOR_YAMLPATH "/etc/opt/nec/ve/mmm/info/ve_hw_spec.yaml"
#define AURORA3_SENSOR_YAMLPATH "/etc/opt/nec/ve3/mmm/info/ve3_hw_spec.yaml"


char path[MAX_LEN];

char *vecmd_bindir = NULL, *veos_bindir = NULL, *sensor_yamlpath = NULL;

#define MAX_VE  8
struct ve_sensor {
    char name[MAX_LEN];
    char sysfs_file[MAX_LEN];
    struct ve_sensor *next;
};

struct ve_yaml {
    char ve[MAX_LEN];          /* veX_Y */
    struct ve_sensor *sensors;
};

struct ve_yaml *ve_yaml;

int ve_count = 0;
int ve_max = MAX_VE;

int
readYaml(const char *file)
{
    FILE *fh;
    yaml_parser_t parser;
    yaml_event_t  event;   /* New variable */
    int doc_lvl = 0;
    int sensor_start = 0;
    int ve_start = 0;
    int search_start = 0;
    int searching = 0;
    struct ve_sensor *vp, *lp;

    fh = fopen(file, "r");
    if(fh == NULL){
        zabbix_log(LOG_LEVEL_DEBUG, "foepn error");
        return(-1);
    }

    /* Initialize parser */
    if(!yaml_parser_initialize(&parser)){
        fclose(fh);
        return(-1);
    }

    /* Set input file */
    yaml_parser_set_input_file(&parser, fh);

    if (ve_yaml == NULL) {
        ve_yaml = zbx_malloc (NULL, sizeof (struct ve_yaml) * ve_max);
        if (ve_yaml == NULL) {
            zabbix_log(LOG_LEVEL_DEBUG, "malloc error");
            return(-1);
        }
    }

    /* START */
    lp = NULL;
    do {
        if (!yaml_parser_parse(&parser, &event)) {
            zabbix_log(LOG_LEVEL_DEBUG, "parser_parse error");
            return(-1);
        }

        switch(event.type){ 
        case YAML_NO_EVENT:
            break;
        /* Stream start/end */
        case YAML_STREAM_START_EVENT:
            doc_lvl = -1;
            sensor_start = 0;
            ve_start = 0;
            search_start = 0;
            break;
        case YAML_STREAM_END_EVENT:
            doc_lvl = -1;
            sensor_start = 0;
            ve_start = 0;
            search_start = 0;
            break;
        /* Block delimeters */
        case YAML_DOCUMENT_START_EVENT:
            doc_lvl = 0;
            sensor_start = 0;
            ve_start = 0;
            search_start = 0;
            break;
        case YAML_DOCUMENT_END_EVENT:
            doc_lvl = -1;
            sensor_start = 0;
            ve_start = 0;
            search_start = 0;
            break;
        case YAML_MAPPING_START_EVENT:
            doc_lvl++;
            if(doc_lvl == MAP_SENSOR){
                vp = NULL;
            }
            break;
        case YAML_MAPPING_END_EVENT:
            doc_lvl--;
            if(doc_lvl <= MAP_TOP){
                sensor_start = 0;
                ve_start = 0;
            }
            if(doc_lvl == MAP_SENSOR){
                if(vp != NULL){
                    if(vp->sysfs_file[0] == '\0'){
                        free(vp);
                    }else{
                        if(ve_yaml[ve_count-1].sensors == NULL){
                            ve_yaml[ve_count-1].sensors = vp;
                        }else{
                            lp->next = vp;
                        }
                        lp = vp;
                    }
                }
            }
            search_start = 0;
            break;
        case YAML_SCALAR_EVENT:
            if(searching){
                zbx_snprintf(vp->sysfs_file, MAX_LEN, "%s", event.data.scalar.value);
                searching = 0;
                break;
            }
            if((doc_lvl == MAP_TOP) && (strcmp((const char *)event.data.scalar.value, SENSOR_INFO) == 0)){
                sensor_start = 1;
            }else if((doc_lvl == MAP_VE) && (sensor_start == 1)){
                if(ve_count >= ve_max){
                    struct ve_yaml *tmp;
                    ve_max += MAX_VE;
                    tmp = (struct ve_yaml *)zbx_realloc(ve_yaml, sizeof(struct ve_yaml) * ve_max);
                    if (tmp == NULL) {
                        free(ve_yaml);
                        ve_yaml = NULL;
                        zabbix_log(LOG_LEVEL_DEBUG, "realloc error");
                        return(-1);
                    }
                    ve_yaml = tmp;
                }
                zbx_snprintf(ve_yaml[ve_count].ve, MAX_LEN, "%s", event.data.scalar.value);
                ve_yaml[ve_count].sensors = NULL;
                ve_count++;
                ve_start = 1;
            }else if((doc_lvl == MAP_SENSOR) && (ve_start == 1)){
                vp = (struct ve_sensor *)zbx_malloc(NULL, sizeof(struct ve_sensor));
                if(vp == NULL){
                    break;
                }
                zbx_snprintf(vp->name, MAX_LEN, "%s", event.data.scalar.value);
                vp->sysfs_file[0] = '\0';
                vp->next = NULL;
                search_start = 1;
            }else if((doc_lvl == MAP_SEARCH) && (search_start == 1)){
                if(strcmp((const char *)event.data.scalar.value, SYSFS_KEY) == 0){
                    searching = 1;
                }
            }else if(doc_lvl <= MAP_SENSOR){
                search_start = 0;
                if(doc_lvl <= MAP_VE){
                    ve_start = 0;
                    if(doc_lvl <= MAP_TOP){
                        sensor_start = 0;
                    }
                }
            }
            break;
        case YAML_SEQUENCE_START_EVENT:
        case YAML_SEQUENCE_END_EVENT:
        case YAML_ALIAS_EVENT:
        default:
            searching = 0;
            break;
        }
        if(event.type != YAML_STREAM_END_EVENT){
            yaml_event_delete(&event);
        }
    } while(event.type != YAML_STREAM_END_EVENT);

    yaml_event_delete(&event);
    /* Cleanup */
    yaml_parser_delete(&parser);
    fclose(fh);

    return(0);
}

char *
getSysFileName(const char *item, const char *ve)
{
    int i;
    struct ve_sensor *p;

    for(i = 0; i < ve_count; i++){
        if(strcmp(ve_yaml[i].ve, ve) != 0){
            continue;
        }
        for(p = ve_yaml[i].sensors; p != NULL; p = p->next){
            if(strcmp(p->name, item) == 0){
                return(p->sysfs_file);
            }
        }
        return(NULL);
    }
    return(NULL);
}

void
prsensor()
{
    int i;
    struct ve_sensor *p;

    for(i = 0; i < ve_count; i++){
        zabbix_log(LOG_LEVEL_DEBUG, "[%s]", ve_yaml[i].ve);
        for(p = ve_yaml[i].sensors; p != NULL; p = p->next){
            zabbix_log (LOG_LEVEL_DEBUG, "  %s: %s", p->name, p->sysfs_file);
        }
    }
}

/******* CONFING ******/
int parse_config_file ()
{
    zbx_cfg_line_t cfg[] = 
    {
        {"VECMD_BINDIR",    &vecmd_bindir,      ZBX_CFG_TYPE_STRING,    ZBX_CONF_PARM_OPT, 0, 0}, 
        {"VEOS_BINDIR",     &veos_bindir,       ZBX_CFG_TYPE_STRING,    ZBX_CONF_PARM_OPT, 0, 0}, 
        {"SENSOR_YAMLPATH", &sensor_yamlpath,   ZBX_CFG_TYPE_STRING,    ZBX_CONF_PARM_OPT, 0, 0}, 
        {NULL}
    };

    /* Parse Config File */
    zbx_parse_cfg_file(CONFIG, cfg, ZBX_CFG_FILE_OPTIONAL, ZBX_CFG_NOT_STRICT, 0);

    /* Set Default Value */
    if (vecmd_bindir == NULL) {
        vecmd_bindir = DEFAULT_VECMD_BINDIR;
    }

    if (veos_bindir == NULL) {
        veos_bindir = DEFAULT_VEOS_BINDIR;
    }

    if (sensor_yamlpath == NULL) {
        sensor_yamlpath = AURORA3_SENSOR_YAMLPATH;
    }

    return 0;
}
