/*
** Zabbix
** Copyright (C) 2001-2016 Zabbix SIA
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

#include "zbxsysinc.h"
#include "module.h"
#include "zbxcommon.h"
#include "zbxlog.h"
#include "zbxcfg.h"
#include "zbxexec.h"
#include "ve_item.h"
#include <linux/limits.h>

extern int readYaml(const char *);
extern void prsensor(void);
extern char *getSysFileName(const char *, const char *);
extern int parse_config_file(void);

/* the variable keeps timeout setting for item processing */
static int item_timeout = 0;

#define VECMD "vecmd"

extern char *vecmd_bindir, *veos_bindir, *sensor_yamlpath;

ve_path_t ve_path[VE_MAX] =
{
    {"veslot0", {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}, "", ""},
    {"veslot1", {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}, "", ""},
    {"veslot2", {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}, "", ""},
    {"veslot3", {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}, "", ""},
    {"veslot4", {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}, "", ""},
    {"veslot5", {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}, "", ""},
    {"veslot6", {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}, "", ""},
    {"veslot7", {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}, "", ""},
};

int zbx_vhmodule_vecmd_items(AGENT_REQUEST *request, AGENT_RESULT *result);
int zbx_vemodule_vecmd_items(AGENT_REQUEST *request, AGENT_RESULT *result);
int zbx_vemodule_sysfs_items(AGENT_REQUEST *request, AGENT_RESULT *result);
int zbx_vemodule_sensor_items(AGENT_REQUEST *request, AGENT_RESULT *result);
int zbx_vemodule_location_item(AGENT_REQUEST *request, AGENT_RESULT *result);

static ZBX_METRIC keys[] =
/*      KEY                             FLAG              FUNCTION                  TEST PARAMETERS */
{
    {"vehw.attached_ves",               CF_HAVEPARAMS,    zbx_vhmodule_vecmd_items, "aurora-VH0000"},
    {"vehw.mmm_version",                CF_HAVEPARAMS,    zbx_vhmodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {"vehw.ve_state",                   CF_HAVEPARAMS,    zbx_vemodule_sysfs_items, "AuroraVE-0000-0, 0"},
    {"vehw.os_state",                   CF_HAVEPARAMS,    zbx_vemodule_sysfs_items, "AuroraVE-0000-0, 0"},
    {"vehw.ve_model",                   CF_HAVEPARAMS,    zbx_vemodule_sysfs_items, "AuroraVE-0000-0, 0"},
    {"vehw.ve_type",                    CF_HAVEPARAMS,    zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {"vehw.product_type",               CF_HAVEPARAMS,    zbx_vemodule_sysfs_items, "AuroraVE-0000-0, 0"},
    {"vehw.cores",                      CF_HAVEPARAMS,    zbx_vemodule_sysfs_items, "AuroraVE-0000-0, 0"},
    {"vehw.ve_chip_sn",                 CF_HAVEPARAMS,    zbx_vemodule_sysfs_items, "AuroraVE-0000-0, 0"},
    {"vehw.ve_board_sn",                CF_HAVEPARAMS,    zbx_vemodule_sysfs_items, "AuroraVE-0000-0, 0"},
    {"vehw.ve_board_name",              CF_HAVEPARAMS,    zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {"vehw.ve_board_parameter",         CF_HAVEPARAMS,    zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {"vehw.ve_driver_version",          CF_HAVEPARAMS,    zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {"vehw.vmcfw_version",              CF_HAVEPARAMS,    zbx_vemodule_sysfs_items, "AuroraVE-0000-0, 0"},
    {"vehw.core_clock",                 CF_HAVEPARAMS,    zbx_vemodule_sysfs_items, "AuroraVE-0000-0, 0"},
    {"vehw.base_clock",                 CF_HAVEPARAMS,    zbx_vemodule_sysfs_items, "AuroraVE-0000-0, 0"},
    {"vehw.memory_clock",               CF_HAVEPARAMS,    zbx_vemodule_sysfs_items, "AuroraVE-0000-0, 0"},
    {"vehw.memory_size",                CF_HAVEPARAMS,    zbx_vemodule_sysfs_items, "AuroraVE-0000-0, 0"},
    {"vehw.l1_icache_size",             CF_HAVEPARAMS,    zbx_vemodule_sysfs_items, "AuroraVE-0000-0, 0"},
    {"vehw.l1_dcache_size",             CF_HAVEPARAMS,    zbx_vemodule_sysfs_items, "AuroraVE-0000-0, 0"},
    {"vehw.l2_cache_size",              CF_HAVEPARAMS,    zbx_vemodule_sysfs_items, "AuroraVE-0000-0, 0"},
    {"vehw.l3_cache_size",              CF_HAVEPARAMS,    zbx_vemodule_sysfs_items, "AuroraVE-0000-0, 0"},
    {"vehw.llc_cache_size",             CF_HAVEPARAMS,    zbx_vemodule_sysfs_items, "AuroraVE-0000-0, 0"},
    {"vehw.partitioning_mode",          CF_HAVEPARAMS,    zbx_vemodule_sysfs_items, "AuroraVE-0000-0, 0"},
    {"vehw.numa_node0",                 CF_HAVEPARAMS,    zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {"vehw.numa_node1",                 CF_HAVEPARAMS,    zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {"vehw.throttling_level",           CF_HAVEPARAMS,    zbx_vemodule_sysfs_items, "AuroraVE-0000-0, 0"},

    {"vehw.pci.bridge_id",              CF_HAVEPARAMS,    zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {"vehw.pci.bus_id",                 CF_HAVEPARAMS,    zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {"vehw.pci.vendor_id",              CF_HAVEPARAMS,    zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {"vehw.pci.device_id",              CF_HAVEPARAMS,    zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {"vehw.pci.class_code",             CF_HAVEPARAMS,    zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {"vehw.pci.current_link_speed",     CF_HAVEPARAMS,    zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {"vehw.pci.max_link_speed",         CF_HAVEPARAMS,    zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {"vehw.pci.negotiated_link_width",  CF_HAVEPARAMS,    zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {"vehw.pci.maximum_link_width",     CF_HAVEPARAMS,    zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},

    {"vehw.temperature.chip_location",  CF_HAVEPARAMS,    zbx_vemodule_location_item, "AuroraVE-0000-0, 0, 0"},
    {"vehw.temperature.hbm_0",          CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.temperature.hbm_1",          CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.temperature.hbm_2",          CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.temperature.hbm_3",          CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.temperature.hbm_4",          CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.temperature.hbm_5",          CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.temperature.diode0",         CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.temperature.diode1",         CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.temperature.diode2",         CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.temperature.diode3",         CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.temperature.inlet_side",     CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.temperature.outlet_side",    CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.temperature.vrm",            CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},

    {"vehw.voltage.vddh",               CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.voltage.avddh",              CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.voltage.hbm_vdd0",           CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.voltage.hbm_vdd1",           CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.voltage.hbm_vdd2",           CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.voltage.hbm_vpp0",           CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.voltage.hbm_vpp1",           CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.voltage.hbm_vpp2",           CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.voltage.aux_12v_v",          CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.voltage.edge_12v_v",         CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.voltage.edge_3_3v",          CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.voltage.core_vdd",           CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.voltage.pll_vdd",            CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.voltage.avdd",               CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},

    {"vehw.current.aux_12v",            CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},
    {"vehw.current.edge_12v",           CF_HAVEPARAMS,    zbx_vemodule_sensor_items,"AuroraVE-0000-0, 0"},

    {"vehw.eccs.mcu_correctable_error", CF_HAVEPARAMS,    zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {"vehw.eccs.llc_correctable_error", CF_HAVEPARAMS,    zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {"vehw.eccs.core_dp_correctable_error", CF_HAVEPARAMS,zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {"vehw.eccs.comn_dp_correctable_error", CF_HAVEPARAMS,zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {"vehw.eccs.vr_correctable_error",  CF_HAVEPARAMS,    zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {"vehw.eccs.ltc_correctable_error", CF_HAVEPARAMS,    zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {"vehw.eccs.vcc_correctable_error", CF_HAVEPARAMS,    zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {"vehw.errs.fault_occur_time",      CF_HAVEPARAMS,    zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {"vehw.errs.fault_errors",          CF_HAVEPARAMS,    zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {"vehw.errs.non_fault_occur_time",  CF_HAVEPARAMS,    zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {"vehw.errs.non_fault_errors",      CF_HAVEPARAMS,    zbx_vemodule_vecmd_items, "AuroraVE-0000-0, 0"},
    {NULL}
};

static VAL_KFT_t val_kft_vecmd[] =
{
    /*  KEY                             grep key word           value type */
    {"vehw.attached_ves",               "Attached VEs",         AR_UINT64},
    {"vehw.mmm_version",                "MMM Version",          AR_STRING},
    {"vehw.ve_type",                    "VE Type",              AR_STRING},
    {"vehw.ve_board_name",              "VE Board Name",        AR_STRING},
    {"vehw.ve_board_parameter",         "VE Board Parameter",   AR_STRING},
    {"vehw.ve_driver_version",          "VE Driver Version",    AR_STRING},
    {"vehw.numa_node0",                 "NUMA node0",           AR_STRING},
    {"vehw.numa_node1",                 "NUMA node1",           AR_STRING},

    {"vehw.pci.bridge_id",              "Bridge ID",            AR_STRING},
    {"vehw.pci.bus_id",                 "Bus ID",               AR_STRING},
    {"vehw.pci.vendor_id",              "Vendor ID",            AR_STRING},
    {"vehw.pci.device_id",              "Device ID",            AR_STRING},
    {"vehw.pci.class_code",             "Class Code",           AR_STRING},
    {"vehw.pci.current_link_speed",     "Current Link Speed",   AR_DOUBLE},
    {"vehw.pci.max_link_speed",         "Max Link Speed",       AR_DOUBLE},
    {"vehw.pci.negotiated_link_width",  "Negotiated Link Width",AR_STRING},
    {"vehw.pci.maximum_link_width",     "Maximum Link Width",   AR_STRING},

    {"vehw.eccs.mcu_correctable_error", "MCU Correctable Error",AR_STRING},
    {"vehw.eccs.llc_correctable_error", "LLC Correctable Error",AR_STRING},
    {"vehw.eccs.core_dp_correctable_error", "CORE_DP Correctable Error", AR_STRING},
    {"vehw.eccs.comn_dp_correctable_error", "COMN_DP Correctable Error", AR_STRING},
    {"vehw.eccs.vr_correctable_error",  "VR Correctable Error", AR_STRING},
    {"vehw.eccs.ltc_correctable_error", "LTC Correctable Error", AR_STRING},
    {"vehw.eccs.vcc_correctable_error", "VCC Correctable Error", AR_STRING},
    {"vehw.errs.fault_occur_time",      "Fault Occur Time",     AR_STRING},
    {"vehw.errs.fault_errors",          "Fault Errors",         AR_STRING},
    {"vehw.errs.non_fault_occur_time",  "Non Fault Occur Time", AR_STRING},
    {"vehw.errs.non_fault_errors",      "Non Fault Errors", AR_STRING},
    {NULL}
};

static VAL_KFT_t val_kft_sysfs[] =
{
    /*  KEY                     sysfs filename      value type */
    {"vehw.ve_state",           "ve_state",         AR_STRING},
    {"vehw.os_state",           "os_state",         AR_STRING},
    {"vehw.ve_model",           "model",            AR_STRING},
    {"vehw.product_type",       "type",             AR_STRING},
    {"vehw.cores",              "num_of_core",      AR_UINT64},
    {"vehw.ve_chip_sn",         "chip_id",          AR_STRING},
    {"vehw.ve_board_sn",        "serial",           AR_ASCII},
    {"vehw.vmcfw_version",      "fw_version",       AR_VERSION},
    {"vehw.core_clock",         "clock_chip",       AR_UINT64M},
    {"vehw.base_clock",         "clock_base",       AR_UINT64M},
    {"vehw.memory_clock",       "clock_memory",     AR_UINT64M},
    {"vehw.memory_size",        "memory_size",      AR_UINT64},
    {"vehw.l1_icache_size",     "cache_l1i",        AR_UINT64Ki},
    {"vehw.l1_dcache_size",     "cache_l1d",        AR_UINT64Ki},
    {"vehw.l2_cache_size",      "cache_l2",         AR_UINT64Ki},
    {"vehw.l3_cache_size",      "cache_l3",         AR_UINT64Ki},
    {"vehw.llc_cache_size",     "cache_llc",        AR_UINT64Ki},
    {"vehw.partitioning_mode",  "partitioning_mode",AR_STRING},
    {"vehw.throttling_level",   "throttling_level", AR_UINT64},
    {NULL}
};

static VAL_KFT_t val_kft_sensor[] =
{
    /*  KEY                             sensor name             value type */
    {"vehw.temperature.hbm_0",          "ve_hbm0_temp",         AR_UINT64}, 
    {"vehw.temperature.hbm_1",          "ve_hbm1_temp",         AR_UINT64}, 
    {"vehw.temperature.hbm_2",          "ve_hbm2_temp",         AR_UINT64}, 
    {"vehw.temperature.hbm_3",          "ve_hbm3_temp",         AR_UINT64}, 
    {"vehw.temperature.hbm_4",          "ve_hbm4_temp",         AR_UINT64}, 
    {"vehw.temperature.hbm_5",          "ve_hbm5_temp",         AR_UINT64}, 
    {"vehw.temperature.diode0",         "ve_temp_ve_diode_0",   AR_DOUBLE6},
    {"vehw.temperature.diode1",         "ve_temp_ve_diode_1",   AR_DOUBLE6},
    {"vehw.temperature.diode2",         "ve_temp_ve_diode_2",   AR_DOUBLE6},
    {"vehw.temperature.diode3",         "ve_temp_ve_diode_3",   AR_DOUBLE6},
    {"vehw.temperature.inlet_side",     "ve_temp_inlet",   	AR_DOUBLE6},
    {"vehw.temperature.outlet_side",    "ve_temp_outlet",   	AR_DOUBLE6},
    {"vehw.temperature.vrm",    	"ve_temp_vrm",   	AR_DOUBLE6},
    {"vehw.voltage.vddh",               "ve_vddh",              AR_DOUBLE6},
    {"vehw.voltage.avddh",              "ve_avddh",             AR_DOUBLE6}, 
    {"vehw.voltage.avdd",               "ve_avdd",              AR_DOUBLE6},
    {"vehw.voltage.hbm_vdd1",           "ve_hbm_vdd1",          AR_DOUBLE6},
    {"vehw.voltage.hbm_vdd2",           "ve_hbm_vdd2",          AR_DOUBLE6},
    {"vehw.voltage.hbm_vpp2",           "ve_hbm_vpp2",          AR_DOUBLE6},
    {"vehw.voltage.hbm_vdd0",           "ve_hbm_vdd0",          AR_DOUBLE6},
    {"vehw.voltage.hbm_vpp0",           "ve_hbm_vpp0",          AR_DOUBLE6},
    {"vehw.voltage.hbm_vpp1",           "ve_hbm_vpp1",          AR_DOUBLE6},
    {"vehw.voltage.aux_12v_v",          "ve_aux12v",            AR_DOUBLE6},
    {"vehw.voltage.edge_12v_v",         "ve_power_edge_12v",    AR_DOUBLE6},
    {"vehw.voltage.edge_3_3v",          "ve_power_edge_3.3v",   AR_DOUBLE6},
    {"vehw.voltage.core_vdd",           "ve_vdd",               AR_DOUBLE6},
    {"vehw.voltage.pll_vdd",            "ve_pll_vdd",           AR_DOUBLE6},
    {"vehw.current.aux_12v",            "ve_current_0",         AR_DOUBLE3},
    {"vehw.current.edge_12v",           "ve_current_1",         AR_DOUBLE3},
    {NULL}
};

static VAL_KFT_t val_kft_location[] =
{
    /*  KEY                         sensor name         value type */
    {"vehw.temperature.chip_location",  "ve_chip_loc%d_temp",   AR_DOUBLE6},
    {NULL}
};

static int core_num[] = {0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, -1};

/******************************************************************************
 *                                                                            *
 * Function: zbx_module_api_version                                           *
 *                                                                            *
 * Purpose: returns version number of the module interface                    *
 *                                                                            *
 * Return value: ZBX_MODULE_API_VERSION_ONE - the only version supported by   *
 *               Zabbix currently                                             *
 *                                                                            *
 ******************************************************************************/
int zbx_module_api_version()
{
    return ZBX_MODULE_API_VERSION_ONE;
}

/******************************************************************************
 *                                                                            *
 * Function: zbx_module_item_timeout                                          *
 *                                                                            *
 * Purpose: set timeout value for processing of items                         *
 *                                                                            *
 * Parameters: timeout - timeout in seconds, 0 - no timeout set               *
 *                                                                            *
 ******************************************************************************/
void zbx_module_item_timeout(int timeout)
{
    item_timeout = timeout;
}

/******************************************************************************
 *                                                                            *
 * Function: zbx_module_item_list                                             *
 *                                                                            *
 * Purpose: returns list of item keys supported by the module                 *
 *                                                                            *
 * Return value: list of item keys                                            *
 *                                                                            *
 ******************************************************************************/
ZBX_METRIC *zbx_module_item_list()
{
    return keys;
}

static char *get_ve_model (char *vnp)
{
    char *p, *dp;
    char valpath[PATH_MAX];
    char valbuf_model[BUFSIZ], valbuf_product[BUFSIZ];
    VAL_KFT_t *kftp;
    FILE *vfd;
    int vn;

    vn = strtoul(vnp, &dp, 0);
    if (dp == vnp
     || *dp != '\0'
     || vn >= VE_MAX) {
        return NULL;
    }

    if (ve_path[vn].VE_model[0] != '\0') {
        return ve_path[vn].VE_model;
    }

    /* Get model */
    for (kftp = val_kft_sysfs; kftp->key != NULL; kftp++) {
        if (strcmp("vehw.ve_model", kftp->key) == 0)
            break;
    }
    if (kftp == NULL) {
        return NULL;
    }

    zbx_snprintf(valpath, PATH_MAX, "%s/%s", ve_path[vn].path, kftp->f_name);

    vfd = fopen(valpath, "r");
    if (vfd == NULL) {
        return NULL;
    }

    if (fgets(valbuf_model, BUFSIZ, vfd) == NULL) {
        fclose(vfd);
        return NULL;
    }
    fclose(vfd);

    /* Get product */
    for (kftp = val_kft_sysfs; kftp->key != NULL; kftp++) {
        if (strcmp("vehw.product_type", kftp->key) == 0)
            break;
    }
    if (kftp == NULL) {
        return NULL;
    }

    zbx_snprintf(valpath, PATH_MAX, "%s/%s", ve_path[vn].path, kftp->f_name);

    vfd = fopen(valpath, "r");
    if (vfd == NULL) {
        return NULL;
    }

    if (fgets(valbuf_product, BUFSIZ, vfd) == NULL) {
        fclose(vfd);
        return NULL;
    }
    fclose(vfd);

    if ((p = strchr (valbuf_model, '\n')) != NULL) {
        *p = '\0';
    }
    if ((p = strchr (valbuf_product, '\n')) != NULL) {
        *p = '\0';
    }
    zbx_snprintf(ve_path[vn].VE_model, VE_MODEL_MAX, "ve%s_%s", valbuf_model, valbuf_product);

    return ve_path[vn].VE_model;
}

static char hex_bcnt[16] = {0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4};

static void set_zabbix_result (AGENT_RESULT *result, char *valbuf, int type)
{
    int64_t val, cnt;
    double wval;
    char *dp, *stp;
    size_t l;

    switch (type) {
        case AR_UINT64:
            val = strtol(valbuf, &dp, 0);
            SET_UI64_RESULT(result, val);
            break;
        case AR_UINT64Ki:
            val = strtol(valbuf, &dp, 0);
            val *= 0x400;
            SET_UI64_RESULT(result, val);
            break;
        case AR_UINT64K:
            val = strtol(valbuf, &dp, 0);
            val *= 1000;
            SET_UI64_RESULT(result, val);
            break;
        case AR_UINT64K3:
            val = strtol(valbuf, &dp, 0);
            val /= 1000;
            SET_UI64_RESULT(result, val);
            break;
        case AR_UINT64Mi:
            val = strtol(valbuf, &dp, 0);
            val *= 0x100000;
            SET_UI64_RESULT(result, val);
            break;
        case AR_UINT64M:
            val = strtol(valbuf, &dp, 0);
            val *= 1000000;
            SET_UI64_RESULT(result, val);
            break;
        case AR_UINT64G:
            val = strtol(valbuf, &dp, 0);
            val *= 0x40000000;
            SET_UI64_RESULT(result, val);
            break;
        case AR_DOUBLE:
            wval = strtod(valbuf, &dp);
            SET_DBL_RESULT(result, wval);
            break;
        case AR_DOUBLE3:
            val = strtol(valbuf, &dp, 0);
            wval = (double)val / 1000.0;
            SET_DBL_RESULT(result, wval);
            break;
        case AR_DOUBLE6:
            val = strtol(valbuf, &dp, 0);
            wval = (double)val / 1000000.0;
            SET_DBL_RESULT(result, wval);
            break;
        case AR_STRING:
            l = strlen(valbuf);
            if (valbuf[l - 1] == '\n') {
                valbuf[--l] = '\0';
            }
            SET_STR_RESULT(result, strdup(valbuf));
            break;
        case AR_BITCNT:
            val = strtol(valbuf, &dp, 0);
            cnt = 0;
            while (val > 0) {
                int    v;

                v = val & 0x0f;
                cnt += hex_bcnt[v];
                val >>= 4;
            }
            SET_UI64_RESULT(result, cnt);
            break;
        case AR_STATE:
            switch (valbuf[0]) {
                case '0':
                    stp = "UNINITIALIZED";
                    break;
                case '1':
                    stp = "ONLINE";
                    break;
                case '2':
                    stp = "OFFLINE";
                    break;
                case '3':
                    stp = "MAINTENANCE";
                    break;
                case '4':
                    stp = "UNAVAILABLE";
                    break;
                default:
                    stp = "UNKNOWN";
                    break;
            }
            SET_STR_RESULT(result, strdup(stp));
            break;
        case AR_VERSION:
            {
                char str[256];
                int major, minor, release;

                val = strtol (valbuf, &dp, 0);
                major = (val & 0x00F000) >> 12;
                minor = (val & 0x000F00) >>  8;
                release = val & 0x0000FF;
                
                zbx_snprintf (str, sizeof (str), "%x.%x.%x", major, minor, release);
                SET_STR_RESULT(result, strdup(str));
            }
            break;
        case AR_ASCII:
            {
                char str[25], *p, c[3];
                int i = 0;

                p = valbuf;

                memset (c, 0, sizeof (c));

                while ((c[0] != '0' && c[1] != '0') || (*p != '\0' && *p != '\n')) {
                    c[0] = *p++; c[1] = *p++;
                    str[i++] = (char)strtoul(c, NULL, 16);
                    if (i == (sizeof (str) / sizeof (char))) {
                        break;
                    }
                }
                str[i] = '\0';
                SET_STR_RESULT(result, strdup(str));
            }
            break;
    }
}

static char *
zbx_vhmodule_get_vepath(char *vnp, AGENT_RESULT *result)
{
    int vn, rv, l;
    char *dp;
    char cmdbuf[BUFSIZ], errbuf[BUFSIZ];
    char *obp = NULL;
    unsigned char flag = ZBX_EXIT_CODE_CHECKS_ENABLED;

    /* there is no strict validation of parameters for simplicity sake */
    vn = strtoul(vnp, &dp, 0);
    if (dp == vnp
     || *dp != '\0'
     || vn >= VE_MAX) {
        SET_MSG_RESULT(result, strdup("Invalid value given"));
        return(NULL);
    }
    if (ve_path[vn].path[0] != '\0') {
        return(ve_path[vn].path);
    }
    if (ve_path[vn].slot[0] == '\0') {
        SET_MSG_RESULT(result, strdup("device node not found"));
        return(NULL);
    }
    zbx_snprintf(cmdbuf, BUFSIZ, "/usr/bin/udevadm info --query=path --name=%s 2>/dev/null", ve_path[vn].slot);
    rv = zbx_execute(cmdbuf, &obp, errbuf, BUFSIZ, item_timeout, flag, NULL);
    switch (rv) {
        case SUCCEED:
            zabbix_log(LOG_LEVEL_DEBUG, "obp %s", obp);
            if (*obp != '/') {
                SET_MSG_RESULT(result, strdup(obp));
                return(NULL);
            }
            l = strlen(obp);
            if (obp[l - 1] == '\n') {
                obp[--l] = '\0';
            }
            zbx_snprintf(ve_path[vn].path, PATH_MAX, "/sys%s", obp);
            zbx_free(obp);
            break;
        case TIMEOUT_ERROR:
            SET_MSG_RESULT(result, strdup("udev execution timeout"));
            return(NULL);
        case FAIL:
            zabbix_log(LOG_LEVEL_DEBUG, "%s", errbuf);
            SET_MSG_RESULT(result, strdup(errbuf));
            return(NULL);
    }

    return(ve_path[vn].path);
}

static int execute_vecmd(char *vnp, VAL_KFT_t *kftp, AGENT_RESULT *result)
{
    int rv;
    char cmdbuf[BUFSIZ], errbuf[BUFSIZ], *outbp = NULL, *vp;
    unsigned char flag = ZBX_EXIT_CODE_CHECKS_ENABLED;

    if (vnp == NULL) {
        zbx_snprintf(cmdbuf, BUFSIZ, "%s/%s info 2>/dev/null | grep \"%s\"", vecmd_bindir, VECMD, kftp->f_name);
    } else {
        zbx_snprintf(cmdbuf, BUFSIZ, "%s/%s -N %s info 2>/dev/null | grep \"%s\"", vecmd_bindir, VECMD, vnp, kftp->f_name);
    }

    /* Execute vmcmd command */
    rv = zbx_execute(cmdbuf, &outbp, errbuf, BUFSIZ, item_timeout, flag, NULL);
    switch (rv) {
        case SUCCEED:
            /* Get Result */
            strtok(outbp, ":");
            /* vehw.eccs and vehw.errs include space charactor. */
            if (strstr (kftp->key, "vehw.eccs") != NULL || strstr (kftp->key, "vehw.errs") != NULL) {
                vp = strtok(NULL, "\n");
                if (vp == NULL) {
                    SET_MSG_RESULT(result, strdup("Invalid format of vecmd output."));
                    return SYSINFO_RET_FAIL;
                }
                while (*vp == ' ' && *vp != '\0') {
                    vp++;
                }
            } else {
                vp = strtok(NULL, " \n");
            }
            break;
        case TIMEOUT_ERROR:
            SET_MSG_RESULT(result, strdup("vecmd execution timeout"));
            return SYSINFO_RET_FAIL;
        case FAIL:
            zabbix_log(LOG_LEVEL_DEBUG, "%s", errbuf);
            SET_MSG_RESULT(result, strdup(errbuf));
            return SYSINFO_RET_FAIL;
    }

    if (vp == NULL) {
        SET_MSG_RESULT(result, strdup("MMM Service is unavailable."));
        zbx_free(outbp);
        return SYSINFO_RET_FAIL;
    } else {
        set_zabbix_result (result, vp, kftp->r_type);
    }

    zbx_free(outbp);

    return SYSINFO_RET_OK;
}

/******************************************************************************
 *                                                                            *
 * Function: zbx_vhmodule_vecmd_items                                         *
 *                                                                            *
 * Purpose: a main entry point for processing of an item                      *
 *                                                                            *
 * Parameters: request - structure that contains item key and parameters      *
 *              request->key - item key without parameters                    *
 *              request->nparam - number of parameters                        *
 *              request->timeout - processing should not take longer than     *
 *                                 this number of seconds                     *
 *              request->params[N-1] - pointers to item key parameters        *
 *                                                                            *
 *             result - structure that will contain result                    *
 *                                                                            *
 * Return value: SYSINFO_RET_FAIL - function failed, item will be marked      *
 *                                 as not supported by zabbix                 *
 *               SYSINFO_RET_OK - success                                     *
 *                                                                            *
 * Comment: get_rparam(request, N-1) can be used to get a pointer to the Nth  *
 *          parameter starting from 0 (first parameter). Make sure it exists  *
 *          by checking value of request->nparam.                             *
 *                                                                            *
 ******************************************************************************/

int zbx_vhmodule_vecmd_items(AGENT_REQUEST *request, AGENT_RESULT *result)
{
    VAL_KFT_t *kftp;
    char *kp;

    if (request->nparam != 0) {
        /* set optional error message */
        SET_MSG_RESULT(result, strdup("Invalid number of parameters"));
        return SYSINFO_RET_FAIL;
    }

    kp = get_rkey(request);
    for (kftp = val_kft_vecmd; kftp->key != NULL; kftp++) {
        if (strcmp(kp, kftp->key) == 0)
            break;
    }
    if (kftp->key == NULL) {
        /* set optional error message */
        SET_MSG_RESULT(result, strdup("Unknown key name"));
        return SYSINFO_RET_FAIL;
    }

    return execute_vecmd (NULL, kftp, result);
}

int zbx_vemodule_vecmd_items(AGENT_REQUEST *request, AGENT_RESULT *result)
{
    VAL_KFT_t *kftp;
    char *kp, *vnp;

    if (request->nparam != 1) {
        /* set optional error message */
        SET_MSG_RESULT(result, strdup("Invalid number of parameters"));
        return SYSINFO_RET_FAIL;
    }

    kp = get_rkey(request);
    for (kftp = val_kft_vecmd; kftp->key != NULL; kftp++) {
        if (strcmp(kp, kftp->key) == 0)
            break;
    }
    if (kftp->key == NULL) {
        /* set optional error message */
        SET_MSG_RESULT(result, strdup("Unknown key name"));
        return SYSINFO_RET_FAIL;
    }

    /* Create vmcmd command line */
    vnp = get_rparam(request, 0);

    return execute_vecmd (vnp, kftp, result);
}

int zbx_vemodule_sysfs_items(AGENT_REQUEST *request, AGENT_RESULT *result)
{
    char *vnp, *kp, *vpp;
    VAL_KFT_t *kftp;
    FILE *vfd;
    char valpath[PATH_MAX], valbuf[BUFSIZ];

    if (request->nparam != 1) {
        /* set optional error message */
        SET_MSG_RESULT(result, strdup("Invalid number of parameters"));
        return SYSINFO_RET_FAIL;
    }

    kp = get_rkey(request);
    for (kftp = val_kft_sysfs; kftp->key != NULL; kftp++) {
        if (strcmp(kp, kftp->key) == 0)
            break;
    }
    if (kftp->key == NULL) {
        /* set optional error message */
        SET_MSG_RESULT(result, strdup("Unknown key name"));
        return SYSINFO_RET_FAIL;
    }

    vnp = get_rparam(request, 0);
    vpp = zbx_vhmodule_get_vepath(vnp, result);
    if (vpp == NULL) {
        return SYSINFO_RET_FAIL;
    }

    zbx_snprintf(valpath, PATH_MAX, "%s/%s", vpp, kftp->f_name);
    zabbix_log(LOG_LEVEL_DEBUG, "file=%s", valpath);

    vfd = fopen(valpath, "r");
    if (vfd == NULL) {
        SET_MSG_RESULT(result, strdup("File can not access"));
        *vpp = '\0';
        return SYSINFO_RET_FAIL;
    }

    if (fgets(valbuf, BUFSIZ, vfd) == NULL) {
        SET_MSG_RESULT(result, strdup("File can not access"));
        fclose(vfd);
        return SYSINFO_RET_FAIL;
    }

    fclose(vfd);

    set_zabbix_result (result, valbuf, kftp->r_type);

    return SYSINFO_RET_OK;
}

int zbx_vemodule_sensor_items(AGENT_REQUEST *request, AGENT_RESULT *result)
{
    char *vnp, *kp, *vpp, *model_p;
    VAL_KFT_t *kftp;
    FILE *vfd;
    char valpath[PATH_MAX], valbuf[BUFSIZ];

    if (request->nparam != 1) {
        /* set optional error message */
        SET_MSG_RESULT(result, strdup("Invalid number of parameters"));
        return SYSINFO_RET_FAIL;
    }

    kp = get_rkey(request);
    for (kftp = val_kft_sensor; kftp->key != NULL; kftp++) {
        if (strcmp(kp, kftp->key) == 0) {
            break;
        }
    }
    if (kftp->key == NULL) {
        /* set optional error message */
        SET_MSG_RESULT(result, strdup("Unknown key name"));
        return SYSINFO_RET_FAIL;
    }

    vnp = get_rparam(request, 0);
    vpp = zbx_vhmodule_get_vepath(vnp, result);
    if (vpp == NULL) {
        return SYSINFO_RET_FAIL;
    }

    if ((model_p = get_ve_model (vnp)) == NULL) {
        SET_MSG_RESULT(result, strdup("Failed to get ve_model_product"));
        return SYSINFO_RET_FAIL;
    }

    zbx_snprintf(valpath, PATH_MAX, "%s/%s", vpp, getSysFileName(kftp->f_name, model_p));
    zabbix_log(LOG_LEVEL_DEBUG, "file=%s", valpath);

    vfd = fopen(valpath, "r");
    if (vfd == NULL) {
        SET_MSG_RESULT(result, strdup("File can not access"));
        *vpp = '\0';
        return SYSINFO_RET_FAIL;
    }
    if (fgets(valbuf, BUFSIZ, vfd) == NULL) {
        SET_MSG_RESULT(result, strdup("File can not access"));
        fclose(vfd);
        return SYSINFO_RET_FAIL;
    }
    fclose(vfd);

    set_zabbix_result (result, valbuf, kftp->r_type);

    return SYSINFO_RET_OK;
}

static int
ve_get_physcore(int ven, char *dirp, int lcn, AGENT_RESULT *result)
{
    u_int64_t enab;
    int l, p;
    char valpath[BUFSIZ], valbuf[BUFSIZ], *dp;
    FILE *vfd;

    if (ve_path[ven].lpcn[0] >= 0) {
        zabbix_log(LOG_LEVEL_DEBUG, "ve=%d, lcn=%d pcn=%d", ven, lcn, ve_path[ven].lpcn[lcn]);
        return(ve_path[ven].lpcn[lcn]);
    }
    zbx_snprintf(valpath, BUFSIZ, "%s/cores_enable", dirp);
    zabbix_log(LOG_LEVEL_DEBUG, "file=%s", valpath);

    vfd = fopen(valpath, "r");
    if (vfd == NULL) {
        SET_MSG_RESULT(result, strdup("File(cores_enable) can not access"));
        return -2;
    }
    if (fgets(valbuf, BUFSIZ, vfd) == NULL) {
        SET_MSG_RESULT(result, strdup("File(cores_enable) can not access"));
        fclose(vfd);
        return -2;
    }
    fclose(vfd);

    enab = strtoul(valbuf, &dp, 0);
    if (enab == 0) {
        SET_MSG_RESULT(result, strdup("cores_enable is invalid"));
        return -2;
    }
    for (p = 0, l = 0; p < VE_CMAX && enab != 0; p++, enab >>= 1) {
        if ((enab & 0x01) == 1) {
            zabbix_log(LOG_LEVEL_DEBUG, "ve=%d, L=%d P=%d", ven, l, p);
            ve_path[ven].lpcn[l++] = p;
        }
    }
    for ( ; l < VE_CMAX; l++) {
        ve_path[ven].lpcn[l] = -1;
    }

    zabbix_log(LOG_LEVEL_DEBUG, "ve=%d, lcn=%d pcn=%d", ven, lcn, ve_path[ven].lpcn[lcn]);

    return(ve_path[ven].lpcn[lcn]);
}

int zbx_vemodule_location_item(AGENT_REQUEST *request, AGENT_RESULT *result)
{
    char *vnp, *cnp, *kp, *vpp, *fp, *dp, *model_p;
    int vn, cn, lcn, l;
    VAL_KFT_t *kftp;
    FILE *vfd;
    char valpath[PATH_MAX], valbuf[BUFSIZ], filbuf[32];

    if (request->nparam != 2) {
        /* set optional error message */
        SET_MSG_RESULT(result, strdup("Invalid number of parameters"));
        return SYSINFO_RET_FAIL;
    }

    kp = get_rkey(request);

    for (kftp = val_kft_location; kftp->key != NULL; kftp++) {
        if (strcmp(kp, kftp->key) == 0) {
            break;
        }
    }
    if (kftp->key == NULL) {
        /* set optional error message */
        SET_MSG_RESULT(result, strdup("Unknown key name"));
        return SYSINFO_RET_FAIL;
    }

    vnp = get_rparam(request, 0);
    vpp = zbx_vhmodule_get_vepath(vnp, result);
    if (vpp == NULL) {
        return SYSINFO_RET_FAIL;
    }
    vn = strtol(vnp, &dp, 0);

    zbx_snprintf(valpath, PATH_MAX, "%s", vpp);

    cnp = get_rparam(request, 1);

    /* there is no strict validation of parameters for simplicity sake */
    lcn = strtoul(cnp, &dp, 0);
    if (dp == cnp
     || *dp != '\0')
    {
        SET_MSG_RESULT(result, strdup("Invalid value given"));
        return SYSINFO_RET_FAIL;
    }

    cn = ve_get_physcore(vn, valpath, lcn, result);
    if (cn < 0) {
        if (cn == -1) {
            SET_MSG_RESULT(result, strdup("Specified core is not available"));
        }
        return SYSINFO_RET_FAIL;
    }
    for (fp = kftp->f_name, l = 0; *fp != '\0'; ) {
        if (*fp != '%') {
            filbuf[l++] = *fp++;
            continue;
        }
        fp++;
        switch (*fp) {
            case 'd':
                zbx_snprintf(&filbuf[l], (32 - l), "%d", core_num[lcn]); /* for Aurora3 */
                l += strlen(&filbuf[l]);
                break;
            default:
                filbuf[l++] = '%';
                filbuf[l++] = *fp;
                break;
        }
        fp++;
    }
    filbuf[l] = '\0';

    if ((model_p = get_ve_model(vnp)) == NULL) {
        SET_MSG_RESULT(result, strdup("Failed to get ve_model_product"));
        return SYSINFO_RET_FAIL;
    }

    zbx_snprintf (valpath, sizeof (valpath), "%s/%s", vpp, getSysFileName(filbuf, model_p));
    zabbix_log(LOG_LEVEL_DEBUG, "File=%s", valpath);

    vfd = fopen(valpath, "r");
    if (vfd == NULL) {
        SET_MSG_RESULT(result, strdup("File can not access"));
        *vpp = '\0';
        return SYSINFO_RET_FAIL;
    }
    if (fgets(valbuf, BUFSIZ, vfd) == NULL) {
        SET_MSG_RESULT(result, strdup("File can not access"));
        fclose(vfd);
        ve_path[vn].lpcn[0] = -1;
        return SYSINFO_RET_FAIL;
    }
    fclose(vfd);

    set_zabbix_result (result, valbuf, kftp->r_type);

    return SYSINFO_RET_OK;
}

/******************************************************************************
 *                                                                            *
 * Function: zbx_module_init                                                  *
 *                                                                            *
 * Purpose: the function is called on agent startup                           *
 *          It should be used to call any initialization routines             *
 *                                                                            *
 * Return value: ZBX_MODULE_OK - success                                      *
 *               ZBX_MODULE_FAIL - module initialization failed               *
 *                                                                            *
 * Comment: the module won't be loaded in case of ZBX_MODULE_FAIL             *
 *                                                                            *
 ******************************************************************************/
int zbx_module_init()
{
    parse_config_file();
    zabbix_log(LOG_LEVEL_DEBUG, "ve_hw_item init.");
    zabbix_log(LOG_LEVEL_DEBUG, "vecmd: %s, veos: %s, sensor: %s", vecmd_bindir, veos_bindir, sensor_yamlpath);

    /* Read VEHW yaml File */
    if (readYaml (sensor_yamlpath) < 0) {
        return ZBX_MODULE_FAIL;
    }

    /* Logging Sensor info */
    prsensor();

    return ZBX_MODULE_OK;
}

/******************************************************************************
 *                                                                            *
 * Function: zbx_module_uninit                                                *
 *                                                                            *
 * Purpose: the function is called on agent shutdown                          *
 *          It should be used to cleanup used resources if there are any      *
 *                                                                            *
 * Return value: ZBX_MODULE_OK - success                                      *
 *               ZBX_MODULE_FAIL - function failed                            *
 *                                                                            *
 ******************************************************************************/
int zbx_module_uninit()
{
    zabbix_log(LOG_LEVEL_DEBUG, "ve_hw_item uninit.");

    return ZBX_MODULE_OK;
}
